#------------------------------------------------------------------------------------------------------------------------------------
# URBAN FORESTRY SOUTH ECOSHAPE TOOL
#
#
# AUTHOR:	Johnny Reed
# DATE:		July 26, 2010  Last modified March 15, 2011
# PURPOSE:	This script is designed to be run as a script tool in an ArcGIS toolbox. It first validates and then imports tree inventory
# data from a shapefile into an Access database created by the i-Tree Eco program, populating the appropriate database tables with the
# data and converting the database into an ESRI personal geodatabase. Spatial data and user-defined attributes from the shapefile are stored
# in a feature class within the geodatabase. The database can subsequently be submitted to the USDA Forest Service for analysis using the
# i-Tree Eco program and the results returned to the user.
#
# This script was written using Python 2.5.1 and ArcGIS 9.3.1. It may not work with earlier versions.
#--------------------------------------------------------------------------------------------------------------------------------------

# Import required modules
import sys, os, arcgisscripting, datetime, logging


# Create the geoprocessor
gp = arcgisscripting.create(9.3)


# Enable the overwrite option
gp.OverwriteOutput = 1


# Load data management toolbox
# Load data management toolbox
if gp.Exists("C:\Program Files (x86)\ArcGIS\Desktop10.0\ArcToolbox\Toolboxes "):
     ## 64 bit
     gp.AddToolbox("C:\Program Files (x86)\ArcGIS\Desktop10.0\ArcToolbox\Toolboxes\Data Management Tools.tbx")
     gp.AddToolbox("C:\Program Files (x86)\ArcGIS\Desktop10.0\ArcToolbox\Toolboxes\Conversion Tools.tbx")
     GIS_System = "64 bit - ArcMap v10"
elif gp.Exists("C:\Program Files\ArcGIS\Desktop10.0\ArcToolbox\Toolboxes"):
     ## 64 bit
     gp.AddToolbox("C:\Program Files\ArcGIS\Desktop10.0\ArcToolbox\Toolboxes\Data Management Tools.tbx")
     gp.AddToolbox("C:\Program Files\ArcGIS\Desktop10.0\ArcToolbox\Toolboxes\Conversion Tools.tbx")
     GIS_System = "64 bit - ArcMap v10"
elif gp.Exists("C:\Program Files (x86)\ArcGIS\ArcToolbox\Toolboxes"):
     ## 64 bit
     gp.AddToolbox("C:\Program Files (x86)\ArcGIS\ArcToolbox\Toolboxes\Data Management Tools.tbx")
     gp.AddToolbox("C:\Program Files (x86)\ArcGIS\ArcToolbox\Toolboxes\Conversion Tools.tbx")
     GIS_System = "64 bit - ArcMap v9.3"
else:
     ## 32 bit
     gp.AddToolbox("C:\Program Files\ArcGIS\ArcToolbox\Toolboxes\Data Management Tools.tbx")
     gp.AddToolbox("C:\Program Files\ArcGIS\ArcToolbox\Toolboxes\Conversion Tools.tbx")
     GIS_System = "32 bit - ArcMap v9.3"

# Script parameters
prj_fldr = gp.GetParameterAsText(0)			## project folder (workspace)
eco_mdb = gp.GetParameterAsText(1)			## i-Tree Eco database (workspace)
backup_mdb = gp.GetParameterAsText(2)  		## backup for i-Tree Eco database (workspace)
gps_shp = gp.GetParameterAsText(3)			## GPS shapefile (shapefile)
land_use = gp.GetParameterAsText(4)			## LandUse code (string)
specieslist = gp.GetParameterAsText(5)		## SpeciesList (table)
genustable = gp.GetParameterAsText(6)		## GenusTable (table)

gp.workspace = prj_fldr

now = datetime.datetime.now()


# Open text file for log
log = open(prj_fldr + "/EcoShapeLog.txt", 'w')
log.write("\nECOSHAPE LOG")
log.write("\n\n" + str(now))

log.write("\n\n\nGIS System:  " + GIS_System)
log.write("\n\n\nProject Folder:  " + prj_fldr)
log.write("\ni-Tree Eco Database:  " + eco_mdb)
log.write("\nDatabase Backup:  " + backup_mdb)
log.write("\nGPS Shapefile:  " + gps_shp)
log.write("\nLandUse Code:  " + land_use)
log.write("\nSpeciesList:  " + specieslist)
log.write("\nGenusTable:  " + genustable)


# Define function to allow for both display and logging of messages
def AddMsgAndLog(msg, severity):
    try:
        log.write("\n" + msg)
        if severity == 0:
            gp.AddMessage(msg)
        elif severity == 1:
            gp.AddWarning(msg)
        elif severity ==2:
            gp.AddError(msg)
		
    except:
        gp.AddError(gp.GetMessages(2))
        log.write("\n\n" + gp.GetMessages(2))


AddMsgAndLog("\n\nLog file created at " + prj_fldr + "/EcoShapeLog.txt", 0)

####################################################
### Make an archival copy of i-Tree Eco database ###
####################################################

AddMsgAndLog("\n\nCREATING A BACKUP COPY OF THE ECO DATABASE", 0)
AddMsgAndLog("\nA copy of " + eco_mdb + " will be saved to:  " + backup_mdb, 0)

try:
    gp.Copy_management(eco_mdb, backup_mdb)
    AddMsgAndLog("\n" + backup_mdb + " created.", 0)
    	
except:
    AddMsgAndLog("\nCreation of backup failed. Backup MDB could not be created.\n", 1)
    #AddMsgAndLog(gp.GetMessages(2), 2)
    

	
#########################################
### Verify structure of GPS shapefile ###
#########################################

AddMsgAndLog("\n\nVERIFYING THE STRUCTURE OF THE GPS SHAPEFILE", 0)

shp_errors = []
field_names = []

field_list = gp.ListFields(gps_shp)
field_names = [field.name for field in field_list]


# Check for presence of minimum required attributes

AddMsgAndLog("\nChecking " + gps_shp + " for minimum required attributes...", 0)


min_req_att = ["TREEID", "SPECIES", "DBH1", "DBH2", "DBH3", "DBH4", "DBH5", "DBH6", "TREE_HT", "CRWN_BASE", "CRWN_WD1", "CRWN_WD2", "MISSING", "DIEBACK", "CLE", "COMMENTS"]

not_exist = []

try:
    for req_att in min_req_att:
        if req_att in field_names:
            AddMsgAndLog(req_att + " exists", 0)
        else:
            AddMsgAndLog(req_att + " does not exist", 2)
            not_exist.append(req_att)
            
    if not_exist == []:
        AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " contains all of the minimum required attribute fields.", 0)
    else:        		
        shp_errors.append("Minimum Required Attributes Error")
        raise "MinReqAttError"
    
except "MinReqAttError":
    AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " is missing one or more minimum required attributes. These errors must be corrected before the shapefile can be processed.\n\nThe following required attributes are missing from the GPS shapefile:\n" + str(not_exist) + "\n", 2)
	
	
# Check for attribute names with leading underscore "_"

AddMsgAndLog("\nChecking " + gps_shp + " for attribute names with a leading underscore \"_\"...", 0)

leading = []

try:
    for field in field_list:
        if str(field.name)[0] == "_":
            leading.append(str(field.name))
			
    if leading == []:
        AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " contains no attribute names with a leading underscore \"_\".", 0)
    else:
        shp_errors.append("Leading Underscore Error")
        raise "LeadingUnderscoreError"
	
except "LeadingUnderscoreError":
    AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " contains one or more attribute names with a leading underscore \"_\". These errors must be corrected before the shapefile can be processed.\n\nThe following attribute names contain a leading underscore \"_\":\n" + str(leading) + "\n", 2)


# Check for optional attributes and user attributes with a leading "x"

AddMsgAndLog("\nChecking " + gps_shp + " for optional attributes and user attributes with a leading \"x\"...", 0)

opt_att = []

for field in field_list:
    if field.name == "PLOTID":
        opt_att.append(str(field.name))
    elif field.name == "STREET":
        opt_att.append(str(field.name))
    elif str(field.name)[0] == "x":
        opt_att.append(str(field.name))

if not opt_att == []:
    AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " contains the following optional attributes and/or user attributes beginning with a leading \"x\":\n" + str(opt_att), 0)
else:
    AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " contains no optional attributes and/or user attributes beginning with a leading \"x\".", 0)

	
# Validate attribute types

AddMsgAndLog("\nChecking " + gps_shp + " for valid attribute types...", 0)

invalid_type = []
	
try:
    for field in field_list:
        if field.name == "PLOTID":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "TREEID":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "SPECIES":
            if not field.type == "String":
                invalid_type.append(str(field.name))
        elif field.name == "DBH1":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "DBH2":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "DBH3":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "DBH4":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "DBH5":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "DBH6":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "TREE_HT":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "CRWN_BASE":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "CRWN_WD1":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "CRWN_WD2":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "MISSING":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "DIEBACK":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "STREET":
            if not field.type == "String":
                invalid_type.append(str(field.name))
        elif field.name == "CLE":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "COMMENTS":
            if not field.type == "String":
                invalid_type.append(str(field.name))
        elif field.name == "xDiamHt":
            if not (field.type == "Single" or field.type == "Double" or field.type == "Integer" or field.type == "Long" or field.type == "Short" or field.type == "SmallInteger"):
                invalid_type.append(str(field.name))
        elif field.name == "xLandUse":
            if not field.type == "String":
                invalid_type.append(str(field.name))
        elif field.name == "xCrew":
            if not field.type == "String":
                invalid_type.append(str(field.name))
        elif field.name == "xInvDate":
            if not field.type == "Date":
                invalid_type.append(str(field.name))

    if invalid_type == []:
        AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " contains attributes of the correct type.", 0)
    elif not invalid_type == []:        
        shp_errors.append("Attribute Type Error")
        raise "FieldTypeError"
				
except "FieldTypeError":									
    AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " has one or more attributes with an invalid type. These errors must be corrected before the shapefile can be processed.\n\nThe following attributes have an invalid type:\n" + str(invalid_type) + "\n", 2)
	

# Validate attribute values

AddMsgAndLog("\nChecking " + gps_shp + " for valid attribute values...", 0)

def unique_value(inTable, inField):   #function for determining if all values in a given field are unique
    rows = gp.searchcursor(inTable, "", "", inField)
    row = rows.next()
    unique = True
    uniqueList = []
    while row:
        if not row.GetValue(inField) in uniqueList:
            uniqueList.append(row.GetValue(inField))
        elif row.GetValue(inField) in uniqueList:
            unique = False		
        row = rows.next()
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows
    return unique		

	
def dup_values(inTable, nonuniqueList):    #funtion for identifying nonunique values in a specified field
    values = []
    unique = []
    dups = []
    for field in nonuniqueList:
        rows = gp.SearchCursor(inTable, "", "", field)
        row = rows.next()
        while row:
            values.append(row.GetValue(field))
            if not row.GetValue(field) in unique:
                unique.append(row.GetValue(field))
            row = rows.next()
        for value in unique:
            count = values.count(value)
            if count > 1:
                dups.append([field, value, count])
        if 'row' in dir():
            del row
        if 'rows' in dir():
            del rows
    return dups    


not_unique = []
	
try:
    for field in field_list:
        if field.name == "PLOTID":
            if not unique_value(gps_shp, field.name):
                not_unique.append(str(field.name))
        elif field.name == "TREEID":
            if not unique_value(gps_shp, field.name):
                not_unique.append(str(field.name))

    if not_unique ==[]:
        AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " contains unique values for TREEID and/or PLOTID.", 0)

    elif not not_unique == []:
        duplicates = dup_values(gps_shp, not_unique)
        shp_errors.append("Unique Value Error")
        raise "UniqueValueError"
	
except "UniqueValueError":
    AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " has one or more minimum required attributes which require unique values, but contain nonunique values. These errors must be corrected before the shapefile can be processed.\n\nThe following nonunique values were found [Attribute, Duplicated Value, Number of Occurrences]:\n" + str(duplicates) + "\n", 2)


# Create list of valid species codes using values from the SpeciesList and GenusTable tables
AddMsgAndLog("\nCreating a list of valid species codes.", 0)

gp.workspace = eco_mdb

valid_species = []

try:
    if gp.Exists("SpeciesList"):
        AddMsgAndLog("\n'SpeciesList' already exists in the database and will be deleted.", 0)
        gp.Delete("SpeciesList")
        AddMsgAndLog("\n'SpeciesList' deleted.", 0)
    if gp.Exists("GenusTable"):
        AddMsgAndLog("\n'GenusTable' already exists in the database and will be deleted.", 0)
        gp.Delete("GenusTable")
        AddMsgAndLog("\n'GenusTable' deleted.", 0)
		
except:
    AddMsgAndLog("\nDeletion of 'SpeciesList' and 'GenusTable' failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
    sys.exit()

	
try:
    AddMsgAndLog("\nCopying the tables 'SpeciesList' and 'GenusTable' to the database...", 0)
    gp.TableToTable_conversion(specieslist, eco_mdb, "SpeciesList")
    gp.TableToTable_conversion(genustable, eco_mdb, "GenusTable")
    AddMsgAndLog("\n'SpeciesList' and 'GenusTable' copied.", 0)
	
except:
    AddMsgAndLog("\nCopying of 'SpeciesList' and 'GenusTable' failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
    sys.exit()
	

try:
    AddMsgAndLog("\nAdding values from 'SpeciesList' to list of valid codes...", 0)
    rows = gp.SearchCursor("SpeciesList", "", "", "SppCode")
    row = rows.Next()
    while row:
        valid_species.append(row.GetValue("SppCode"))
        row = rows.Next()
	
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows
		
    AddMsgAndLog("\nValues from 'SpeciesList' added to list of valid codes.", 0)
	
except:
    AddMsgAndLog("\nAddition of values from 'SpeciesList' to list of valid codes failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
    sys.exit()


try:
    AddMsgAndLog("\nAdding values from 'GenusTable' to list of valid codes...", 0)
    rows = gp.SearchCursor("GenusTable", "", "", "GenusCode")
    row = rows.Next()
    while row:
        valid_species.append(row.GetValue("GenusCode"))
        row = rows.Next()
	
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows

    AddMsgAndLog("\nValues from 'GenusTable' added to list of valid codes.", 0)
	
except:
    AddMsgAndLog("\nAddition of values from 'GenusTable' to list of valid codes failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
    sys.exit()	

	
length = len(valid_species)
AddMsgAndLog("\nNumber of valid species = " + str(length), 0)
if not length >= 100:
    AddMsgAndLog("The number of valid species is less than 100. This indicates there was an error in creating the list.", 2)
    sys.exit()


try:
    AddMsgAndLog("\nDeleting 'SpeciesList' and 'GenusTable' from database...", 0)
    gp.Delete("SpeciesList")
    gp.Delete("GenusTable")
    AddMsgAndLog("\n'SpeciesList' and 'GenusTable' deleted.", 0)
	
except:
    AddMsgAndLog("\nDeletion of 'SpeciesList' and 'GenusTable' failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)




invalid_value = []

try:	
    rows = gp.searchcursor(gps_shp)
    row = rows.next()
    while row:
        for field in field_list:
            if field.name == "PLOTID":
                if not row.GetValue(field.name) >= 1:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "TREEID":
                if not row.GetValue(field.name) >= 1:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "SPECIES":
                if not len(row.GetValue(field.name)) <= 8:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
                elif not row.GetValue(field.name) in valid_species:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "DBH1":
                if not 0.5 <= row.GetValue(field.name) <= 200:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "DBH2":
                if not (0 <= row.GetValue(field.name) <= 200 or row.GetValue(field.name) == -1):
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "DBH3":
                if not (0 <= row.GetValue(field.name) <= 200 or row.GetValue(field.name) == -1):
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "DBH4":
                if not (0 <= row.GetValue(field.name) <= 200 or row.GetValue(field.name) == -1):
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "DBH5":
                if not (0 <= row.GetValue(field.name) <= 200 or row.GetValue(field.name) == -1):
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "DBH6":
                if not (0 <= row.GetValue(field.name) <= 200 or row.GetValue(field.name) == -1):
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "TREE_HT":
                if not 0 < row.GetValue(field.name) <= 150:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "CRWN_BASE":
                if not (0 <= row.GetValue(field.name) <= 150 or row.GetValue(field.name) == -1):
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "CRWN_WD1":
                if not (0 <= row.GetValue(field.name) <= 150 or row.GetValue(field.name) == -1):
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "CRWN_WD2":
                if not (0 <= row.GetValue(field.name) <= 150 or row.GetValue(field.name) == -1):
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "MISSING":
                if not 0 <= row.GetValue(field.name) <= 100:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "DIEBACK":
                if not 0 <= row.GetValue(field.name) <= 100:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "STREET":
                if not len(row.GetValue(field.name)) == 1 and ((row.GetValue(field.name) == "S" or row.GetValue(field.name) == "Y" or row.GetValue(field.name) == "N")):
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "CLE":
                if not 0 <= row.GetValue(field.name) <= 5:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "COMMENTS":
                if not len(row.GetValue(field.name)) <= 255:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "xDiamHt":
                if not 0 < row.GetValue(field.name) <= 6:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "xLandUse":
                if not len(row.GetValue(field.name)) == 1:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
            elif field.name == "xCrew":
                if not len(row.GetValue(field.name)) <= 12:
                    invalid_value.append([row.GetValue("FID"), str(field.name)])
        row = rows.next()	

    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows

    if invalid_value == []:
        AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " contains valid attribute values.", 0)	

    elif not invalid_value == []:
        shp_errors.append("Attribute Value Error")
        raise "AttributeValueError"	
	
except "AttributeValueError":
    AddMsgAndLog("\nThe GPS shapefile " + gps_shp + " contains one or more records with invalid values. These errors must be corrected before the shapefile can be processed.\n\nThe following records contain invalid values [FID, Attribute Name]:\n" + str(invalid_value) + "\n", 2)
	

	
##### GPS Shapefile Error Summary ####

try:
    if shp_errors == []:
        AddMsgAndLog("\nThe shapefile " + gps_shp + " has the proper structure and is ready to be processed.\n", 0)	
    else:
        raise "ShapeFileError"
	
except "ShapeFileError":
    AddMsgAndLog("\n\nThe shapefile " + gps_shp + " contains errors which must be corrected before it can be processed.\nThe following errors were found:", 2)
    if "Minimum Required Attributes Error" in shp_errors:
        AddMsgAndLog("\nThe following required attributes are missing from the GPS shapefile:\n" + str(not_exist), 2)
    if "Leading Underscore Error" in shp_errors:
        AddMsgAndLog("\nThe following attribute names contain a leading underscore \"_\":\n" + str(leading), 2)
    if "Attribute Type Error" in shp_errors:
        AddMsgAndLog("\nThe following attributes have an invalid type:\n" + str(invalid_type), 2)
    if "Unique Value Error" in shp_errors:
        AddMsgAndLog("\nThe following nonunique values were found [Attribute, Duplicated Value, Number of Occurrences]:\n" + str(duplicates), 2)
    if "Attribute Value Error" in shp_errors:
        AddMsgAndLog("\nThe following records contain invalid values [FID, Attribute Name]:\n" + str(invalid_value), 2)
    sys.exit()



	
######################################################################	
## Check for and if necessary add XY coordinate fields to shapefile ##
######################################################################

AddMsgAndLog("\n\nADDING XY COORDINATE FIELDS POINT_X AND POINT_Y TO SHAPEFILE IF BOTH DO NOT ALREADY EXIST", 0)

try:
    if "POINT_X" in field_names and "POINT_Y" in field_names:
        AddMsgAndLog("\nThe fields POINT_X and POINT_Y already exist.", 0)
        x_coord = "POINT_X"
        y_coord = "POINT_Y"
    else:
        AddMsgAndLog("\nThe fields POINT_X and POINT_Y do not exist. Adding XY coordinate fields to " + gps_shp + " ...", 0)
        gp.AddXY_management(gps_shp)
        AddMsgAndLog("\nXY coordinate fields added.", 0)
        x_coord = "POINT_X"
        y_coord = "POINT_Y"

except:
    AddMsgAndLog("\nAddition of XY coordinate fields failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
    sys.exit()
	
	

#########################################################################
## Check number of plots existing in Eco database.                     ##
## If only 1 plot exists, delete it and proceed with transfer of data. ##
## If 0 plots exists, return an error.								   ##
## If > 1 plot exists, continue by appending new data.                 ##
#########################################################################

AddMsgAndLog("\n\nCHECKING FOR EXISTING PLOTS IN THE ECO DATABASE", 0)

AddMsgAndLog("\nChecking " + eco_mdb + " for existing plots...", 0)

gp.workspace = eco_mdb

try:
    result = gp.GetCount_management("Plots")
    plotcount = int(result.GetOutput(0))
    AddMsgAndLog("\nNumber of existing plots in " + eco_mdb + " = " + str(plotcount), 0)
	
except:
    AddMsgAndLog("\nCounting of existing plots failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)

	
try:
    if plotcount == 1:
        AddMsgAndLog("\nDeleting existing plot...", 0)
        gp.DeleteRows("Plots")
        AddMsgAndLog("\nExisting plot deleted.", 0)
    elif plotcount == 0:
        raise "NoExistingPlot"
    else:
        AddMsgAndLog("\nNew data will be appended to existing data.", 0)	

except "NoExistingPlot":
    AddMsgAndLog("\nThe i-Tree Eco database must contain at least one plot.\n\n", 2)
    sys.exit()
		

		

########################################################################
## Check for duplicate TreeID values in GPS Shapefile and Trees table ##
########################################################################

AddMsgAndLog("\n\nCHECKING FOR DUPLICATE TREEID VALUES IN GPS SHAPEFILE AND TREES TABLE", 0)

gp.workspace = eco_mdb

try:
    AddMsgAndLog("\nChecking for duplicate TREEID values in GPS shapefile and Trees table...", 0)
    ID_list = []
    nonuniqueID = []
    rows = gp.SearchCursor("Trees")
    row = rows.Next()
    while row:
        ID_list.append(row.GetValue("TreeID"))
        row = rows.Next()
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows
    
    rows = gp.SearchCursor(gps_shp)
    row = rows.Next()
    while row:
        if row.GetValue("TREEID") in ID_list:
            nonuniqueID.append(row.GetValue("TREEID"))
        row = rows.Next()
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows        
        
    if nonuniqueID == []:
        AddMsgAndLog("\nNo duplicate TREEID values found.", 0)

    if not nonuniqueID == []:
        shp_errors.append("Duplicate TreeID Error")        
        raise "DuplicateTreeIDError"
            
except "DuplicateTreeIDError":
    AddMsgAndLog("\nThe GPS shape file " + gps_shp + " contains one or more records with values for TREEID which are duplicates of TreeID values in the 'Trees' table. TREEID must be a unique value for each record. The data from " + gps_shp + " can not be processed.\n\nThe following TREEID values are duplicates of TreeID values in the 'Trees' table:\n" + str(nonuniqueID), 2)
    sys.exit()

	
	
	
######################################################
## Import data from GPS shapefile into Eco database ##
######################################################

AddMsgAndLog("\n\nCOPYING DATA FROM SHAPEFILE INTO DATABASE", 0)

AddMsgAndLog("\nImporting the GPS shapefile " + gps_shp + " into " + eco_mdb + " as database table 'GPS_TempData'...", 0)

# If GPS_TempData already exists, delete it
try:
    if gp.Exists("GPS_TempData"):
        AddMsgAndLog("\n'GPS_TempData' already exists.\nDeleting 'GPS_TempData'...", 0)
        gp.Delete_management("GPS_TempData")
        AddMsgAndLog("\nExisting 'GPS_TempData' deleted.", 0)

except:
    AddMsgAndLog("\nDeletion of 'GPS_TempData' failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
    sys.exit()
	
# Copy GPS shapefile to GPS_TempData table
try:
    AddMsgAndLog("\nCopying GPS shapefile to 'GPS_TempData'...", 0)
    gp.TableToTable_conversion(gps_shp, eco_mdb, "GPS_TempData")
    AddMsgAndLog("\n'GPS_TempData' created.", 0)
	
except:
    AddMsgAndLog("\nConversion of the shapefile failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
    sys.exit()
		

		
# Add xLandUse field to GPS_TempData if it does not already exist ##
# If necessary set it equal to land_use parameter from dialog     ##

AddMsgAndLog("\n\nADDING LANDUSE FIELD TO GPS DATA IF IT DOES NOT ALREADY EXIST", 0)

try:
    if not "xLandUse" in field_names:
        AddMsgAndLog("\nxLandUse field does not exist. Creating xLandUse field...", 0)
        gp.AddField("GPS_TempData", "xLandUse", "TEXT", "#", "#", 1)
        rows = gp.UpdateCursor("GPS_TempData")
        row = rows.Next()
        while row:
            row.xLandUse = land_use[0]
            rows.UpdateRow(row)
            row = rows.Next()
        AddMsgAndLog("\nxLandUse field added to 'GPS_TempData'", 0)
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows

except:
    AddMsgAndLog("\nAddition of xLandUse field failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
    sys.exit()


### Convert "Y" to "S" in "STREET" ###
### Convert " " to "N" in "STREET" ###

AddMsgAndLog("\n\nCONVERTNG \"Y\" TO \"S\" IN STREET FIELD", 0)

try:
    rows = gp.UpdateCursor("GPS_TempData")
    row = rows.Next()
    while row:
        if row.STREET.strip() == "Y":
            row.STREET = "S"
            rows.UpdateRow(row)
        elif row.STREET.strip() == "":
            row.STREET = "N"
            rows.UpdateRow(row)
        row = rows.Next()
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows
    AddMsgAndLog("\n'Y'converted to 'S'in STREET field.", 0)
		
except:
    AddMsgAndLog("\nConversion of 'Y' to 'S' failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)


	
### Use rounding to convert "MISSING" and "DIEBACK" values to values allowed by Access database ###

try:
    rows = gp.UpdateCursor("GPS_TempData")
    row = rows.Next()
    while row:
        if 0 <= row.MISSING < 1.5:
            row.MISSING = 0
            rows.UpdateRow(row)
        elif 1.5 <= row.MISSING < 5.5:
            row.MISSING = 3
            rows.UpdateRow(row)
        elif 5.5 <= row.MISSING < 10.5:
            row.MISSING = 8
            rows.UpdateRow(row)
        elif 10.5 <= row.MISSING < 15.5:
            row.MISSING = 13
            rows.UpdateRow(row)
        elif 15.5 <= row.MISSING < 20.5:
            row.MISSING = 18
            rows.UpdateRow(row)
        elif 20.5 <= row.MISSING < 25.5:
            row.MISSING = 23
            rows.UpdateRow(row)
        elif 25.5 <= row.MISSING < 30.5:
            row.MISSING = 28
            rows.UpdateRow(row)
        elif 30.5 <= row.MISSING < 35.5:
            row.MISSING = 33
            rows.UpdateRow(row)
        elif 35.5 <= row.MISSING < 40.5:
            row.MISSING = 38
            rows.UpdateRow(row)
        elif 40.5 <= row.MISSING < 45.5:
            row.MISSING = 43
            rows.UpdateRow(row)
        elif 45.5 <= row.MISSING < 50.5:
            row.MISSING = 48
            rows.UpdateRow(row)
        elif 50.5 <= row.MISSING < 55.5:
            row.MISSING = 53
            rows.UpdateRow(row)
        elif 55.5 <= row.MISSING < 60.5:
            row.MISSING = 58
            rows.UpdateRow(row)
        elif 60.5 <= row.MISSING < 65.5:
            row.MISSING = 63
            rows.UpdateRow(row)
        elif 65.5 <= row.MISSING < 70.5:
            row.MISSING = 68
            rows.UpdateRow(row)
        elif 70.5 <= row.MISSING < 75.5:
            row.MISSING = 73
            rows.UpdateRow(row)
        elif 75.5 <= row.MISSING < 80.5:
            row.MISSING = 78
            rows.UpdateRow(row)
        elif 80.5 <= row.MISSING < 85.5:
            row.MISSING = 83
            rows.UpdateRow(row)
        elif 85.5 <= row.MISSING < 90.5:
            row.MISSING = 88
            rows.UpdateRow(row)
        elif 90.5 <= row.MISSING < 95.5:
            row.MISSING = 93
            rows.UpdateRow(row)
        elif 95.5 <= row.MISSING < 99:
            row.MISSING = 98
            rows.UpdateRow(row)
        elif 99 <= row.MISSING <= 100:
            row.MISSING = 100
            rows.UpdateRow(row)
        row = rows.Next()
			
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows
    AddMsgAndLog("\nValues in MISSING field rounded to allowed values.", 0)
		
except:
    AddMsgAndLog("\nRounding of values in MISSING field failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)


	
try:
    rows = gp.UpdateCursor("GPS_TempData")
    row = rows.Next()
    while row:
        if 0 <= row.DIEBACK < 1.5:
            row.DIEBACK = 0
            rows.UpdateRow(row)
        elif 1.5 <= row.DIEBACK < 5.5:
            row.DIEBACK = 3
            rows.UpdateRow(row)
        elif 5.5 <= row.DIEBACK < 10.5:
            row.DIEBACK = 8
            rows.UpdateRow(row)
        elif 10.5 <= row.DIEBACK < 15.5:
            row.DIEBACK = 13
            rows.UpdateRow(row)
        elif 15.5 <= row.DIEBACK < 20.5:
            row.DIEBACK = 18
            rows.UpdateRow(row)
        elif 20.5 <= row.DIEBACK < 25.5:
            row.DIEBACK = 23
            rows.UpdateRow(row)
        elif 25.5 <= row.DIEBACK < 30.5:
            row.DIEBACK = 28
            rows.UpdateRow(row)
        elif 30.5 <= row.DIEBACK < 35.5:
            row.DIEBACK = 33
            rows.UpdateRow(row)
        elif 35.5 <= row.DIEBACK < 40.5:
            row.DIEBACK = 38
            rows.UpdateRow(row)
        elif 40.5 <= row.DIEBACK < 45.5:
            row.DIEBACK = 43
            rows.UpdateRow(row)
        elif 45.5 <= row.DIEBACK < 50.5:
            row.DIEBACK = 48
            rows.UpdateRow(row)
        elif 50.5 <= row.DIEBACK < 55.5:
            row.DIEBACK = 53
            rows.UpdateRow(row)
        elif 55.5 <= row.DIEBACK < 60.5:
            row.DIEBACK = 58
            rows.UpdateRow(row)
        elif 60.5 <= row.DIEBACK < 65.5:
            row.DIEBACK = 63
            rows.UpdateRow(row)
        elif 65.5 <= row.DIEBACK < 70.5:
            row.DIEBACK = 68
            rows.UpdateRow(row)
        elif 70.5 <= row.DIEBACK < 75.5:
            row.DIEBACK = 73
            rows.UpdateRow(row)
        elif 75.5 <= row.DIEBACK < 80.5:
            row.DIEBACK = 78
            rows.UpdateRow(row)
        elif 80.5 <= row.DIEBACK < 85.5:
            row.DIEBACK = 83
            rows.UpdateRow(row)
        elif 85.5 <= row.DIEBACK < 90.5:
            row.DIEBACK = 88
            rows.UpdateRow(row)
        elif 90.5 <= row.DIEBACK < 95.5:
            row.DIEBACK = 93
            rows.UpdateRow(row)
        elif 95.5 <= row.DIEBACK < 99:
            row.DIEBACK = 98
            rows.UpdateRow(row)
        elif 99 <= row.DIEBACK <= 100:
            row.DIEBACK = 100
            rows.UpdateRow(row)
        row = rows.Next()
			
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows
    AddMsgAndLog("\nValues in DIEBACK field rounded to allowed values.", 0)
		
except:
    AddMsgAndLog("\nRounding of values in DIEBACK field failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)


	
	
### Add Eco fields to GPS_TempData ###

AddMsgAndLog("\n\nADDING ECO FIELDS TO GPS DATA", 0)

try:
    AddMsgAndLog("\nAdding fields to 'GPS_TempData'...", 0)
 # LocationName
    gp.AddField("GPS_TempData", "LocationName", "TEXT")
 # Series
    gp.AddField("GPS_TempData", "Series", "TEXT")
 # Year
    gp.AddField("GPS_TempData", "Year1", "SHORT")
 # PlotID
    if "PLOTID" in field_names:
        gp.DeleteField_management("GPS_TempData", "PLOTID")
    gp.AddField("GPS_TempData", "PlotID", "LONG")
 # SubplotID
    gp.AddField("GPS_TempData", "SubPlotID", "LONG")
	
	
    AddMsgAndLog("\nEco fields added to 'GPS_TempData'.", 0)

except:
    AddMsgAndLog("\nAddition of fields to 'GPS_TempData' failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
 
 
## Populate Eco fields in GPS_TempData

AddMsgAndLog("\n\nPOPULATING ECO FIELDS IN GPS DATA", 0)

# Get values for Location, Series, and Year from Year table in Eco database
# Gets values from first row in Year table. Assumes values are the same for all records.

try:
    AddMsgAndLog("\nPopulating Eco fields in 'GPS_TempData'...", 0)
    rows = gp.SearchCursor("Year")
    row = rows.Next()
    location = row.LocationName
    series = row.Series
    year = row.Year
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows

# Apply values to GPS_TempData
    rows = gp.UpdateCursor("GPS_TempData")
    row = rows.Next()

    while row:
        row.LocationName = location
        row.Series = series
        row.Year1 = year
        row.PlotID = row.TreeID
        row.SubplotID = 1
        rows.UpdateRow(row)
        row = rows.Next()
    if 'row' in dir():
        del row
    if 'rows' in dir():
        del rows
    AddMsgAndLog("\nPopulation of Eco fields in 'GPS_TempData' completed.", 0)
		
		
except:
    AddMsgAndLog("\nPopulation of Eco fields in GPS_TempData failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)

	
### Create xTempData table

try:
    AddMsgAndLog("\nChecking for existence of 'xTempData' table...", 0)
    if gp.Exists("xTempData"):
        AddMsgAndLog("\n'xTempData' already exists and will be deleted.", 0)
        gp.Delete("xTempData")
        AddMsgAndLog("\nExisting 'xTempData' deleted.", 0)
    elif not gp.Exists("xTempData"):
        AddMsgAndLog("'xTempData' does not exist and will be created.", 0)
		
except:
    AddMsgAndLog("\nDeletion of 'xTempData' failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
	
try:
    AddMsgAndLog("\nCreating 'xTempData' table...", 0)
    gp.CreateTable(eco_mdb, "xTempData")
    AddMsgAndLog("\n'xTempData' created.", 0)

except:
    AddMsgAndLog("\nCreation of 'xTempData' failed.", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
	

### Add fields to xTempData table

try:
    AddMsgAndLog("\nAdding attribute fields to 'xTempData' table...", 0)
# LocationName, Series, Year, PlotID, SubplotID, TreeID, all user attributes with leading "x", XY coordinates
    gp.AddField("xTempData", "LocationName", "TEXT")
    AddMsgAndLog("LocationName added", 0)
    gp.AddField("xTempData", "Series", "TEXT")
    AddMsgAndLog("Series added", 0)
    gp.AddField("xTempData", "Year1", "SHORT")
    AddMsgAndLog("Year1 added", 0)
    gp.AddField("xTempData", "PlotID", "LONG")
    AddMsgAndLog("PlotID added", 0)
    gp.AddField("xTempData", "SubPlotID", "LONG")
    AddMsgAndLog("SubPlotID added", 0)
    gp.AddField("xTempData", "TreeID", "LONG")
    AddMsgAndLog("TreeID added", 0)

# Modified 21Jan11 to include date and String (drh)
    xfields = []
    for field in field_list:
        name = str(field.name)
        if name[0] == "x":
            if field.type.upper() == "SMALLINTEGER":
                type = "SHORT"
            elif field.type.upper() == "CHAR":
                type = "TEXT"
            elif field.type.upper() == "CHARACTER":
                type = "TEXT"
            elif field.type.upper() == "C":
                type = "TEXT"
            elif field.type.upper() == "STRING":
                type = "TEXT"
            elif field.type.upper() == "TEXT":
                type = "TEXT"
            elif field.type.upper() == "DATE":
                type = "DATE"
            elif field.type.upper() == "LONG":
                type = "LONG"
            elif field.type.upper() == "INTEGER":
                type = "LONG"
            elif field.type.upper() == "SINGLE":
                type = "FLOAT"
            elif field.type.upper() == "DOUBLE":
                type = "DOUBLE"
            else:
                type = "TEXT"
                
# Truncates user attribute field names to 9 characters        
            if len(name) > 9:
                xname = name[:9]
                AddMsgAndLog(name + " truncated to " + xname, 0)
            else:
                xname = name
                
            gp.AddField("xTempData", xname, type)
            xfields.append(xname)
            AddMsgAndLog(xname + " added", 0)
 		
    gp.AddField("xTempData", x_coord, "DOUBLE")
    AddMsgAndLog(x_coord + " added", 0)
    gp.AddField("xTempData", y_coord, "DOUBLE")
    AddMsgAndLog(y_coord + " added", 0)
    AddMsgAndLog("\nAddition of fields to 'xTempData' completed.\n", 0)

except:
    AddMsgAndLog("\nAddition of attribute fields to 'xTempData' failed.\n", 2)
    # AddMsgAndLog(GetMessages(2), 2)
	


######################################################################
### Populate Eco tables and 'xTempData' with data from GPS_TempData ##
######################################################################

AddMsgAndLog("\n\nPOPULATING ECO AND xTEMPDATA TABLES WITH DATA FROM GPS_TEMPDATA", 0)

try:
    AddMsgAndLog("\nPopulating fields...", 0)
	
    tempRows = gp.SearchCursor("GPS_TempData")
    tempRow = tempRows.Next()

    xRows = gp.InsertCursor("xTempData")
    treeRows = gp.InsertCursor("Trees")
    plotRows = gp.InsertCursor("Plots")
    subRows = gp.InsertCursor("Subplots")
    fluRows = gp.InsertCursor("FieldLandUses")
    gcRows = gp.InsertCursor("GroundCovers")
    diamRows = gp.InsertCursor("TreeDiameters")

    rows_processed = 0
    plots_added = 0
    subplots_added = 0
    trees_added = 0
    fieldlanduses_added = 0
    groundcovers_added = 0
    diameters_added = 0
    xTempData_added = 0

    while tempRow:

#xTempData
        xRow = xRows.NewRow()
        xRow.SetValue("LocationName", tempRow.GetValue("LocationName"))
        xRow.SetValue("Series", tempRow.GetValue("Series"))
        xRow.SetValue("Year1", tempRow.GetValue("Year1"))
        xRow.SetValue("PlotID", tempRow.GetValue("TREEID"))     ##### PlotID always = TreeID #####
        xRow.SetValue("SubPlotID", tempRow.GetValue("SubPlotID"))
        xRow.SetValue("TreeID", tempRow.GetValue("TREEID"))
        xRow.SetValue(x_coord, tempRow.GetValue(x_coord))
        xRow.SetValue(y_coord, tempRow.GetValue(y_coord))
                        
        for field in field_list:
            name = str(field.name)
            if name[0] == "x":
                if len(name) > 9:
                    xname = name[:9]
                else:
                    xname = name
                xRow.SetValue(xname, tempRow.GetValue(name))
        
        xRows.InsertRow(xRow)
        xTempData_added += 1
 
	
# Plots
        plotRow = plotRows.NewRow()
        plotRow.SetValue("LocationName", tempRow.GetValue("LocationName"))
        plotRow.SetValue("Series", tempRow.GetValue("Series"))
        plotRow.SetValue("Year", tempRow.GetValue("Year1"))
        plotRow.SetValue("PlotID", tempRow.GetValue("TREEID"))     ##### PlotID always = TreeID #####
        plotRow.SetValue("XCoordinate", -1)
        plotRow.SetValue("YCoordinate", -1)
        plotRow.SetValue("MapLandUseTypeID", 1)
        if "xInvDate" in field_names:
            plotRow.SetValue("Date", tempRow.GetValue("xInvDate"))
        else:
            plotRow.SetValue("Date", "04/04/2011")
        if "xCrew" in field_names:
            plotRow.SetValue("Crew", tempRow.GetValue("xCrew"))
        else:
            plotRow.SetValue("Crew", "EcoShape")        

        plotRows.InsertRow(plotRow)
        plots_added += 1

	
# Subplots
        subRow = subRows.NewRow()
        subRow.SetValue("LocationName", tempRow.GetValue("LocationName"))
        subRow.SetValue("Series", tempRow.GetValue("Series"))
        subRow.SetValue("Year", tempRow.GetValue("Year1"))
        subRow.SetValue("PlotID", tempRow.GetValue("TREEID"))     ##### PlotID always = TreeID #####
        subRow.SetValue("SubPlotID", tempRow.GetValue("SubPlotID"))	
        subRow.SetValue("NextAvailableTreeID", -1)
        subRow.SetValue("SubplotSize", -1)
        subRow.SetValue("OffsetPoint", -1)
        subRow.SetValue("PercentTreeCover", -1)
        subRow.SetValue("PercentShrubCover", -1)
        subRow.SetValue("PercentPlantable", -1)
        subRow.SetValue("PercentMeasured", 100)

        subRows.InsertRow(subRow)
        subplots_added += 1
    

# Trees
        treeRow = treeRows.NewRow()
        treeRow.SetValue("LocationName", tempRow.GetValue("LocationName"))
        treeRow.SetValue("Series", tempRow.GetValue("Series"))
        treeRow.SetValue("Year", tempRow.GetValue("Year1"))
        treeRow.SetValue("PlotID", tempRow.GetValue("TREEID"))     ##### PlotID always = TreeID #####
        treeRow.SetValue("SubPlotID", tempRow.GetValue("SubPlotID"))
        treeRow.SetValue("TreeID", tempRow.GetValue("TREEID"))
        treeRow.SetValue("FieldLandUse", tempRow.GetValue("xLandUse"))
        treeRow.SetValue("DirectionfromCenter", -1)		
        treeRow.SetValue("DistancefromCenter", -1)
        treeRow.SetValue("TreeStatus", "O")
        treeRow.SetValue("Species", tempRow.GetValue("SPECIES"))
        treeRow.SetValue("HeighttoCrownBase", float(tempRow.GetValue("CRWN_BASE")))
        treeRow.SetValue("TreeHeightLiveTop", float(tempRow.GetValue("TREE_HT")))
        treeRow.SetValue("TreeHeightTotal", float(tempRow.GetValue("TREE_HT")))
        treeRow.SetValue("CrownWidth1", float(tempRow.GetValue("CRWN_WD1")))
        treeRow.SetValue("CrownWidth2", float(tempRow.GetValue("CRWN_WD2")))
        treeRow.SetValue("CrownLightExposure", tempRow.GetValue("CLE"))
        treeRow.SetValue("CrownPosition", -1)
        treeRow.SetValue("PercentCrownMissing", tempRow.GetValue("MISSING"))
        treeRow.SetValue("Transparency", -1)
        treeRow.SetValue("CrownDensity", -1)
        treeRow.SetValue("CrownDieback", tempRow.GetValue("DIEBACK"))
        treeRow.SetValue("PercentImperviousBelow", -1)
        treeRow.SetValue("PercentShrubBelow", -1)
        treeRow.SetValue("TreeSite", tempRow.GetValue("STREET").strip())
        treeRow.SetValue("Comments", tempRow.GetValue("COMMENTS"))
		
        treeRows.InsertRow(treeRow)
        trees_added += 1
    	

	# FieldLandUses
        fluRow = fluRows.NewRow()
        fluRow.SetValue("LocationName", tempRow.GetValue("LocationName"))
        fluRow.SetValue("Series", tempRow.GetValue("Series"))
        fluRow.SetValue("Year", tempRow.GetValue("Year1"))
        fluRow.SetValue("PlotID", tempRow.GetValue("TREEID"))     ##### PlotID always = TreeID #####
        fluRow.SetValue("SubPlotID", tempRow.GetValue("SubPlotID"))	
        fluRow.SetValue("FieldLandUse", tempRow.GetValue("xLandUse"))
        fluRow.SetValue("PercentofSubplot", 100)

        fluRows.InsertRow(fluRow)
        fieldlanduses_added += 1
    
	
# GroundCovers
        gcRow = gcRows.NewRow()
        gcRow.SetValue("LocationName", tempRow.GetValue("LocationName"))
        gcRow.SetValue("Series", tempRow.GetValue("Series"))
        gcRow.SetValue("Year", tempRow.GetValue("Year1"))
        gcRow.SetValue("PlotID", tempRow.GetValue("TREEID"))     ##### lotID always = TreeID #####
        gcRow.SetValue("SubPlotID", tempRow.GetValue("SubPlotID"))	
        gcRow.SetValue("CoverID", 8)
        gcRow.SetValue("PercentofSubplot", 100)

        gcRows.InsertRow(gcRow)
        groundcovers_added += 1
	
	
# TreeDiameters
        x = 1
        while x <= 6:
            diam = "DBH" + str(x)
            if 0.5 <= tempRow.GetValue(diam) <= 200:
                diamRow = diamRows.NewRow()
                diamRow.SetValue("LocationName", tempRow.GetValue("LocationName"))
                diamRow.SetValue("Series", tempRow.GetValue("Series"))
                diamRow.SetValue("Year", tempRow.GetValue("Year1"))
                diamRow.SetValue("PlotID", tempRow.GetValue("TREEID"))     ##### PlotID always = TreeID #####
                diamRow.SetValue("SubPlotID", tempRow.GetValue("SubPlotID"))
                diamRow.SetValue("TreeID", tempRow.GetValue("TREEID"))
                diamRow.SetValue("StemID", x)
                diamRow.SetValue("Diameter", tempRow.GetValue(diam))
                if "xDiamHt" in field_names:
                    diamRow.SetValue("DiameterHeight", tempRow.GetValue("xDiamHt"))
                else:
                    diamRow.SetValue("DiameterHeight", 4.5)
                diamRow.SetValue("WasMeasured", 1)
				
                diamRows.InsertRow(diamRow)
                diameters_added += 1
            x += 1
        rows_processed += 1	
        tempRow = tempRows.Next()
	
    if tempRow in dir():
        del tempRow
    if tempRows in dir():
        del tempRows	
    if plotRow in dir():
        del plotRow
    if plotRows in dir():
        del plotRows	
    if subRow in dir():
        del subRow
    if subRows in dir():
        del subRows
    if treeRow in dir():
        del treeRow
    if treeRows in dir():
        del treeRows
    if fluRow in dir():
        del fluRow
    if fluRows in dir():
        del fluRows
    if gcRow in dir():
        del gcRow
    if gcRows in dir():
        del gcRows
    if diamRow in dir():
        del diamRow
    if diamRows in dir():
        del diamRows
    if xRow in dir():
        del xRow
    if xRows in dir():
        del xRows

    AddMsgAndLog("\nPopulation of fields completed.", 0)
	
except:
    AddMsgAndLog("\nPopulation of fields failed.", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)

	
AddMsgAndLog("Number of records processed from GPS_TempData:  " + str(rows_processed), 0)
AddMsgAndLog("Number of records added to xTempData table:  " + str(xTempData_added), 0)
AddMsgAndLog("Number of records added to Plots table:  " + str(plots_added), 0)
AddMsgAndLog("Number of records added to Subplots table:  " + str(subplots_added), 0)
AddMsgAndLog("Number of records added to Trees table:  " + str(trees_added), 0)
AddMsgAndLog("Number of records added to FieldLandUses table:  " + str(fieldlanduses_added), 0)
AddMsgAndLog("Number of records added to GroundCovers table:  " + str(groundcovers_added), 0)
AddMsgAndLog("Number of records added to TreeDiameters table:  " + str(diameters_added), 0)


#######################################
## Create xTempDataFC from xTempData ##
#######################################

AddMsgAndLog("\n\nCREATING xTEMPDATAFC FEATURE CLASS FROM xTEMPDATA TABLE", 0)

# Make XY Event Layer from xTempData

try:
    AddMsgAndLog("\nChecking for existence of 'xTempDataXY event layer...", 0)
    if gp.Exists("xTempDataXY"):
        AddMsgAndLog("\n'xTempDataXY' already exists and will be deleted.", 0)
        gp.Delete("xTempDataXY")
        AddMsgAndLog("\nExisting 'xTempDataXY' deleted.", 0)
    else:
        AddMsgAndLog("\n'xTempDataXY' does not exist and will be created.", 0)
		
except:
    AddMsgAndLog("\nDeletion of 'xTempDataXY' failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)


try:
    AddMsgAndLog("\nCreating 'xTempDataXY' event layer from xTempData table...", 0)
    spatref = gp.Describe(gps_shp).SpatialReference
    gp.MakeXYEventLayer_management("xTempData", x_coord, y_coord, "xTempDataXY", spatref)
    AddMsgAndLog("\n'xTempDataXY' created.", 0)

except:
    AddMsgAndLog("\nCreation of 'xTempDataXY' event layer failed.", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
	

# Copy features to xTempDataFC feature class

try:
    AddMsgAndLog("\nChecking for existence of 'xTempDataFC feature class...", 0)
    if gp.Exists("xTempDataFC"):
        AddMsgAndLog("\n'xTempDataFC' already exists and will be deleted.", 0)
        gp.Delete("xTempDataFC")
        AddMsgAndLog("\nExisting 'xTempDataFC' deleted.", 0)
    else:
        AddMsgAndLog("\n'xTempDataFC' does not exist and will be created.", 0)
		
except:
    AddMsgAndLog("\nDeletion of 'xTempDataFC' failed.\n", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)


try:
    AddMsgAndLog("\nCreating 'xTempDataFC by copying features from 'xTempDataXY' event layer...", 0)
    gp.CopyFeatures_management("xTempDataXY", "xTempDataFC")
    AddMsgAndLog("\n'xTempDataFC' created.", 0)

except:
    AddMsgAndLog("\nCreation of 'xTempDataFC' feature class failed.", 2)
    AddMsgAndLog(gp.GetMessages(2), 2)
	


#################################################################
## Create xTreeData feature class if it does not already exist ##
#################################################################

AddMsgAndLog("\n\nCREATING xTREEDATA FEATURE CLASS IF IT DOES NOT ALREADY EXIST", 0)

gp.workspace = eco_mdb

AddMsgAndLog("\nChecking for existence of 'xTreeData' feature class...", 0)
if not gp.Exists("xTreeData"):
    try:
    ## If xTreeData does not exist, create by copying xTempDataFC
        AddMsgAndLog("\n'xTreeData' does not exist and will be created by copying 'xTempDataFC'.", 0)
        gp.CopyFeatures_management("xTempDataFC", "xTreeData")
        AddMsgAndLog("\n'xTreeData' created.", 0)
		
    except:
        AddMsgAndLog("\nCreation of 'xTreeData' failed.\n", 2)
        AddMsgAndLog(gp.GetMessages(2), 2)

elif gp.Exists("xTreeData"):
    try:
    ## If xTreeData already exists, append rows from xTempDataFC
        AddMsgAndLog("\n'xTreeData' already exists. Data from 'xTempDataFC' will be appended.", 0)
        gp.Append_management("xTempDataFC", "xTreeData")
        AddMsgAndLog("\nData appended.", 0)
        AddMsgAndLog("\nNumber of records added to xTreeData feature class: " + str(xTempData_added), 0)

    except:
        AddMsgAndLog("\nAppending of data failed.\n", 2)
        AddMsgAndLog(gp.GetMessages(2), 2)

# Delete GPS_TempData, xTempDataXY, and xTempDataFC

#if gp.Exists("GPS_TempData"):
#    try:
#        AddMsgAndLog("\nDeleting GPS_TempData...", 0)
#        gp.Delete_management("GPS_TempData")
#        AddMsgAndLog("\n'GPS_TempData' deleted.", 0)
#    except:
#        AddMsgAndLog("\nDeletion of 'GPS_TempData' failed.\n", 2)
#        AddMsgAndLog(gp.GetMessages(2), 2)
#	
#if gp.Exists("xTempData"):
#    try:
#        AddMsgAndLog("\nDeleting xTempData...", 0)
#        gp.Delete_management("xTempData")
#        AddMsgAndLog("\n'xTempData' deleted.", 0)
#    except:
#        AddMsgAndLog("\nDeletion of 'xTempData' failed.\n", 2)
#        AddMsgAndLog(gp.GetMessages(2), 2)
#
#if gp.Exists("xTempDataXY"):
#    try:
#        AddMsgAndLog("\nDeleting xTempDataXY...", 0)
#        gp.Delete_management("xTempDataXY")
#        AddMsgAndLog("\n'xTempDataXY' deleted.", 0)
#    except:
#        AddMsgAndLog("\nDeletion of 'xTempDataXY' failed.\n", 2)
#        AddMsgAndLog(gp.GetMessages(2), 2)
#	
#if gp.Exists("xTempDataFC"):
#    try:
#        AddMsgAndLog("\nDeleting xTempDataFC...", 0)
#        gp.Delete_management("xTempDataFC")
#        AddMsgAndLog("\n'xTempDataFC' deleted.", 0)
#    except:
#        AddMsgAndLog("\nDeletion of 'xTempDataFC' failed.\n", 2)
#        AddMsgAndLog(gp.GetMessages(2), 2)
	
AddMsgAndLog("\n\nECOSHAPE PROCESSING COMPLETED", 0)
	
AddMsgAndLog("\n\nLog file created at " + prj_fldr + "/EcoShapeLog.txt", 0)
log.write("\n\n" + str(now))
	
AddMsgAndLog("\n\nTHE END\n", 0)



